<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;

class ForumCategory extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'slug',
        'description',
        'order',
        'is_active',
        'required_roles',
        'required_permissions',
        'is_public',
    ];

    protected $casts = [
        'required_roles' => 'array',
        'required_permissions' => 'array',
        'is_public' => 'boolean',
    ];

    public function boards(): HasMany
    {
        return $this->hasMany(ForumBoard::class, 'category_id')
            ->orderBy('order');
    }

    /**
     * Get boards that the current user can access
     */
    public function accessibleBoards(): HasMany
    {
        return $this->hasMany(ForumBoard::class, 'category_id')
            ->accessible()
            ->orderBy('order');
    }

    /**
     * Check if the current user can access this category
     */
    public function canAccess($user = null): bool
    {
        $user = $user ?: Auth::user();

        // If category is public, everyone can access
        if ($this->is_public) {
            return true;
        }

        // If no user is logged in and category is not public
        if (!$user) {
            return false;
        }

        // Check required roles
        if (!empty($this->required_roles)) {
            $hasRequiredRole = false;
            foreach ($this->required_roles as $role) {
                if ($user->hasRole($role)) {
                    $hasRequiredRole = true;
                    break;
                }
            }
            if (!$hasRequiredRole) {
                return false;
            }
        }

        // Check required permissions
        if (!empty($this->required_permissions)) {
            foreach ($this->required_permissions as $permission) {
                if (!$user->can($permission)) {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * Scope to filter categories accessible by current user
     */
    public function scopeAccessible(Builder $query)
    {
        $user = Auth::user();

        return $query->where(function ($q) use ($user) {
            // Always include public categories
            $q->where('is_public', true);

            if ($user) {
                // Include categories based on roles
                $q->orWhere(function ($roleQuery) use ($user) {
                    $roleQuery->whereJsonContains('required_roles', $user->getRoleNames()->toArray())
                             ->orWhereNull('required_roles');
                });

                // Include categories based on permissions
                $q->orWhere(function ($permQuery) use ($user) {
                    $userPermissions = $user->getAllPermissions()->pluck('name')->toArray();
                    if (!empty($userPermissions)) {
                        foreach ($userPermissions as $permission) {
                            $permQuery->orWhereJsonContains('required_permissions', $permission);
                        }
                    }
                });
            }
        });
    }
}